/* -------------------------------------------------------
   Hotel Expiry Dashboard — app.js (SAFE / FULL)
   - Waits for DOMContentLoaded
   - Guards all querySelector results (no null .classList)
   - Dark mode toggle (☀️/🌙)
   - Font size adjust (A-/A+)
   - Search, segmented presets (3m/6m/custom dates)
   - Sorting (click headers), pagination
   - CSV export
   - Colored badges + Days Left sticky column
-------------------------------------------------------- */

document.addEventListener('DOMContentLoaded', () => {
  // ====== Config ======
  const API_URL = '../api/hotels.php'; // If your webroot is /public, change to '/api/hotels.php'

  // ====== Elements ======
  const elResults = document.getElementById('results');
  const elTbody   = elResults?.querySelector('tbody') || null;
  const elEmpty   = document.getElementById('empty');
  const elStatus  = document.getElementById('status');

  const elSelectField = document.getElementById('selectField');
  const elSearch      = document.getElementById('searchBox');
  const elBtnSearch   = document.getElementById('btnSearch');

  const segButtons    = Array.from(document.querySelectorAll('.seg-btn'));
  const elCustomDates = document.getElementById('customDates');
  const elFrom        = document.getElementById('fromDate');
  const elTo          = document.getElementById('toDate');

  const elStatTotal   = document.getElementById('statTotal');
  const elStatSoon    = document.getElementById('statSoon');
  const elStatExpired = document.getElementById('statExpired');

  const elPageSize    = document.getElementById('pageSize');
  const elPrev        = document.getElementById('prevPage');
  const elNext        = document.getElementById('nextPage');
  const elPageInfo    = document.getElementById('pageInfo');
  const elPagination  = document.getElementById('pagination');

  const elBtnExport   = document.getElementById('btnExport');
  const elToast       = document.getElementById('toast');

  const elToggleTheme = document.getElementById('toggleTheme');
  const elFontMinus   = document.getElementById('fontMinus');
  const elFontPlus    = document.getElementById('fontPlus');

  // ====== State ======
  let rows = [];
  let filtered = [];
  let sortKey = null;     // e.g. 'days_left' | 'name' | 'unique_id' | 'exp_dt_fire_license'
  let sortDir = 'asc';    // 'asc' | 'desc'
  let currentPage = 1;
  let pageSize = parseInt(elPageSize?.value || '10', 10);
  let baseFont = 16;      // html base font px

  // ====== Helpers ======
  function showToast(msg) {
    if (!elToast) return;
    elToast.textContent = msg;
    elToast.hidden = false;
    setTimeout(() => { elToast.hidden = true; }, 1600);
  }

  function ensureDefaultActiveSegBtn() {
    // If no button has is-active, set the first one (3m) as active
    const active = segButtons.find(b => b.classList.contains('is-active'));
    if (!active && segButtons.length) {
      segButtons[0].classList.add('is-active');
      // Add primary styling if you use Tailwind classes for active state
      segButtons[0].classList.add('bg-primary', 'text-white');
    }
  }

  function activeRange() {
    const btn = segButtons.find(b => b.classList.contains('is-active'));
    return btn?.dataset.range || '3m';
  }

  function parseDate(s) {
    return s ? new Date(s + 'T00:00:00') : null; // local timezone safe
  }

  function daysBetween(a, b) {
    const ms = (b - a) / (1000 * 60 * 60 * 24);
    return Math.floor(ms);
  }

  function computeDaysLeft(row, field) {
    const today = new Date();
    const d = parseDate(row[field]);
    if (!d) return null;
    return daysBetween(today, d);
  }

  function badgeForStatus(status) {
    const s = (status || '').toLowerCase();
    if (s.includes('regular') || s.includes('paid')) return ['bg-emerald-500/15 text-emerald-700 dark:text-emerald-300 border border-emerald-400/30', status];
    if (s.includes('expiring')) return ['bg-yellow-500/15 text-yellow-700 dark:text-yellow-300 border border-yellow-400/30', status];
    if (s.includes('expired') || s.includes('due')) return ['bg-rose-500/15 text-rose-700 dark:text-rose-300 border border-rose-400/30', status];
    return ['bg-gray-300/40 text-gray-700 dark:bg-white/10 dark:text-gray-300 border border-gray-300/60 dark:border-white/15', status || '—'];
  }

  function chip(content, cls) {
    return `<span class="inline-flex items-center rounded-full px-2 py-0.5 text-xs font-semibold ${cls}">${content}</span>`;
  }

  function updateStats(field) {
    if (!elStatTotal || !elStatSoon || !elStatExpired) return;
    const today = new Date();
    let total = filtered.length, soon = 0, expired = 0;
    for (const r of filtered) {
      const dt = parseDate(r[field]);
      if (!dt) continue;
      if (dt < today) expired++;
      else if (daysBetween(today, dt) < 30) soon++;
    }
    elStatTotal.textContent = total;
    elStatSoon.textContent = soon;
    elStatExpired.textContent = expired;
  }

  function applySearchFilter() {
    const q = (elSearch?.value || '').toLowerCase().trim();
    if (!q) return rows.slice();
    return rows.filter(r =>
      (r.name || '').toLowerCase().includes(q) ||
      (r.address || '').toLowerCase().includes(q) ||
      (r.unique_id || '').toLowerCase().includes(q)
    );
  }

  function sortData() {
    if (!sortKey) return;
    const isDate = sortKey.startsWith('exp_dt_') || sortKey === 'tax_payment_upto';
    const isNumeric = sortKey === 'days_left';
    filtered.sort((a, b) => {
      let va, vb;
      if (isNumeric) {
        va = computeDaysLeft(a, elSelectField?.value);
        vb = computeDaysLeft(b, elSelectField?.value);
        va = va ?? 9e9; vb = vb ?? 9e9; // nulls last
      } else if (isDate) {
        va = parseDate(a[sortKey])?.getTime() ?? 0;
        vb = parseDate(b[sortKey])?.getTime() ?? 0;
      } else {
        va = (a[sortKey] || '').toString().toLowerCase();
        vb = (b[sortKey] || '').toString().toLowerCase();
      }
      if (va < vb) return sortDir === 'asc' ? -1 : 1;
      if (va > vb) return sortDir === 'asc' ? 1 : -1;
      return 0;
    });
  }

  function renderTable() {
    if (!elResults || !elTbody) return;

    // Pagination slice
    const start = (currentPage - 1) * pageSize;
    const pageRows = filtered.slice(start, start + pageSize);

    elTbody.innerHTML = '';
    const field = elSelectField?.value || 'exp_dt_fire_license';

    for (const r of pageRows) {
      const dLeft = computeDaysLeft(r, field);
      const isExpired = dLeft !== null && dLeft < 0;
      const isSoon = dLeft !== null && dLeft >= 0 && dLeft < 30;

      const [badgeCls, badgeText] = badgeForStatus(r.tax_status);

      const tr = document.createElement('tr');
      tr.className = [
        'hover:bg-gray-100 dark:hover:bg-white/5 transition-colors',
        isExpired ? 'bg-rose-50 dark:bg-rose-500/5' : isSoon ? 'bg-yellow-50 dark:bg-yellow-500/5' : ''
      ].join(' ');

      tr.innerHTML = `
        <td class="px-3 py-2">${r.unique_id ?? ''}</td>
        <td class="px-3 py-2 font-semibold">${r.name ?? ''}</td>
        <td class="px-3 py-2">${r.address ?? ''}</td>
        <td class="px-3 py-2">${r.exp_dt_trade_license ?? ''}</td>
        <td class="px-3 py-2">${r.exp_dt_food_license ?? ''}</td>
        <td class="px-3 py-2">${r.exp_dt_fire_license ?? ''}</td>
        <td class="px-3 py-2">${r.tax_payment_upto ?? ''}</td>
        <td class="px-3 py-2">${chip(badgeText, badgeCls)}</td>
        <td class="px-3 py-2 sticky-right">
          ${
            dLeft === null ? '—' :
            isExpired ? chip(dLeft, 'bg-rose-500/15 text-rose-700 dark:text-rose-300 border border-rose-400/30') :
            isSoon ? chip(dLeft, 'bg-yellow-500/15 text-yellow-700 dark:text-yellow-300 border border-yellow-400/30') :
                     chip(dLeft, 'bg-emerald-500/15 text-emerald-700 dark:text-emerald-300 border border-emerald-400/30')
          }
        </td>
      `;
      elTbody.appendChild(tr);
    }

    if (elResults) elResults.hidden = pageRows.length === 0;
    if (elEmpty)   elEmpty.hidden   = filtered.length !== 0;
    if (elPagination) elPagination.hidden = filtered.length <= pageSize;

    const totalPages = Math.max(1, Math.ceil(filtered.length / pageSize));
    if (elPageInfo)  elPageInfo.textContent = `Page ${currentPage} / ${totalPages}`;
    if (elPrev)      elPrev.disabled = currentPage <= 1;
    if (elNext)      elNext.disabled = currentPage >= totalPages;

    updateStats(field);
  }

  function refreshAndRender() {
    filtered = applySearchFilter();
    sortData();
    currentPage = 1;
    renderTable();
  }

  // ====== Sorting: attach to header cells safely ======
  const headerKeyMap = [
    'unique_id', 'name', null,
    'exp_dt_trade_license', 'exp_dt_food_license', 'exp_dt_fire_license',
    'tax_payment_upto', null, 'days_left'
  ];

  function initSorting() {
    if (!elResults) return;
    const heads = elResults.querySelectorAll('thead th');
    heads.forEach((th, idx) => {
      const key = headerKeyMap[idx];
      if (!key) return; // non-sortable column
      th.dataset.sort = key;
      th.classList.add('cursor-pointer', 'select-none');
      th.addEventListener('click', () => {
        if (sortKey === key) sortDir = (sortDir === 'asc' ? 'desc' : 'asc');
        else { sortKey = key; sortDir = 'asc'; }
        refreshAndRender();
        if (elStatus) elStatus.textContent = `Sorted by ${key} (${sortDir}).`;
      });
    });
  }

  // ====== Events ======
  // Segmented buttons
  ensureDefaultActiveSegBtn();
  segButtons.forEach(btn => {
    btn.addEventListener('click', () => {
      // Guard: classList exists because btn is an element
      segButtons.forEach(b => {
        // Defensive: only if element exists
        b?.classList.remove('is-active', 'bg-primary', 'text-white');
      });
      btn.classList.add('is-active', 'bg-primary', 'text-white');
      // Guard for #customDates
      if (elCustomDates) {
        const show = btn.dataset.range === 'custom';
        elCustomDates.classList.toggle('hidden', !show);
      }
    });
  });

  // Search input
  elSearch?.addEventListener('input', () => {
    refreshAndRender();
  });

  // Page size
  elPageSize?.addEventListener('change', () => {
    pageSize = parseInt(elPageSize.value, 10);
    currentPage = 1;
    renderTable();
  });

  // Pagination buttons
  elPrev?.addEventListener('click', () => {
    if (currentPage > 1) { currentPage--; renderTable(); }
  });
  elNext?.addEventListener('click', () => {
    const totalPages = Math.ceil(filtered.length / pageSize);
    if (currentPage < totalPages) { currentPage++; renderTable(); }
  });

  // CSV Export
  elBtnExport?.addEventListener('click', () => {
    if (!filtered.length) { showToast('Nothing to export.'); return; }
    const field = elSelectField?.value || 'exp_dt_fire_license';
    const headers = [
      'unique_id','name','address',
      'exp_dt_trade_license','exp_dt_food_license','exp_dt_fire_license',
      'tax_payment_upto','tax_status','days_left'
    ];
    const lines = [headers.join(',')];
    for (const r of filtered) {
      const dLeft = computeDaysLeft(r, field);
      // Simple CSV (replace commas in address). For robust CSV, quote & escape.
      const vals = [
        r.unique_id, r.name, (r.address||'').replace(/\n/g,' ').replace(/,/g,' '),
        r.exp_dt_trade_license, r.exp_dt_food_license, r.exp_dt_fire_license,
        r.tax_payment_upto, r.tax_status, dLeft ?? ''
      ];
      lines.push(vals.map(v => (v==null?'':String(v))).join(','));
    }
    const blob = new Blob([lines.join('\n')], {type:'text/csv;charset=utf-8;'});
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url; a.download = 'hotels.csv';
    document.body.appendChild(a); a.click(); a.remove();
    URL.revokeObjectURL(url);
    showToast('Exported current view to CSV.');
  });

  // Fetch handler (Search)
  elBtnSearch?.addEventListener('click', async () => {
    const field = elSelectField?.value || 'exp_dt_fire_license';
    const range = activeRange();

    const params = new URLSearchParams({ field });
    if (range === '3m' || range === '6m') {
      params.set('range', range);
    } else {
      const from = elFrom?.value?.trim();
      const to   = elTo?.value?.trim();
      if (!from || !to) { showToast('Pick both From and To dates.'); return; }
      params.set('from', from);
      params.set('to', to);
    }

    const url = `${API_URL}?${params.toString()}`;
    if (elStatus) elStatus.textContent = 'Loading…';

    currentPage = 1; sortKey = 'days_left'; sortDir = 'asc';

    try {
      const res = await fetch(url, { headers: { 'Accept': 'application/json' } });
      if (!res.ok) {
        const text = await res.text();
        console.error('[ERR]', res.status, text);
        if (elStatus) elStatus.textContent = `Server error ${res.status}`;
        rows = []; filtered = [];
        renderTable();
        return;
      }

      const data = await res.json();
      // If backend returns an error object
      if (data && !Array.isArray(data) && data.error) {
        console.error('[API ERROR]', data);
        if (elStatus) elStatus.textContent = data.error;
        rows = []; filtered = [];
        renderTable();
        return;
      }

      rows = Array.isArray(data) ? data : [];
      refreshAndRender();

      if (rows.length === 0) {
        if (elStatus) elStatus.textContent = 'No results for these filters.';
      } else {
        if (elStatus) elStatus.textContent = `Found ${rows.length} record(s).`;
      }
    } catch (e) {
      console.error('[FETCH FAIL]', e);
      if (elStatus) elStatus.textContent = 'Failed to load data. Check network/API URL.';
      rows = []; filtered = [];
      renderTable();
    }
  });

  // ====== Theme & Font Size (with localStorage) ======
  function applyBaseFont(px) {
    document.documentElement.style.setProperty('--base-font', `${px}px`);
  }
  function loadFontPref() {
    const saved = parseInt(localStorage.getItem('baseFontPx') || '16', 10);
    baseFont = Number.isFinite(saved) ? saved : 16;
    applyBaseFont(baseFont);
  }
  function saveFontPref() {
    localStorage.setItem('baseFontPx', String(baseFont));
  }

  elFontMinus?.addEventListener('click', () => {
    baseFont = Math.max(12, baseFont - 1);
    applyBaseFont(baseFont);
    saveFontPref();
  });
  elFontPlus?.addEventListener('click', () => {
    baseFont = Math.min(22, baseFont + 1);
    applyBaseFont(baseFont);
    saveFontPref();
  });

  function loadThemePref() {
    const saved = localStorage.getItem('theme'); // 'light' | 'dark' | null
    if (saved === 'dark') {
      document.documentElement.classList.add('dark');
      if (elToggleTheme) elToggleTheme.textContent = '☀️';
    } else {
      document.documentElement.classList.remove('dark');
      if (elToggleTheme) elToggleTheme.textContent = '🌙';
    }
  }
  function saveThemePref() {
    const isDark = document.documentElement.classList.contains('dark');
    localStorage.setItem('theme', isDark ? 'dark' : 'light');
  }

  elToggleTheme?.addEventListener('click', () => {
    document.documentElement.classList.toggle('dark');
    const isDark = document.documentElement.classList.contains('dark');
    if (elToggleTheme) elToggleTheme.textContent = isDark ? '☀️' : '🌙';
    saveThemePref();
    showToast(isDark ? 'Dark mode on' : 'Light mode on');
  });

  // ====== Init ======
  function init() {
    loadThemePref();        // default light unless saved dark
    loadFontPref();         // default 16px unless saved
    initSorting();          // add click-to-sort behavior to headers
    ensureDefaultActiveSegBtn(); // ensure a default active segmented button

    // If #customDates exists, hide unless 'custom' is already active
    const isCustom = activeRange() === 'custom';
    if (elCustomDates) elCustomDates.classList.toggle('hidden', !isCustom);

    if (elStatus) elStatus.textContent = 'Choose filters and press Search.';
  }

  init();
});
